/*
Copyright (C) MOXA Inc. All rights reserved.

This software is distributed under the terms of the
MOXA License.  See the file COPYING-MOXA for details.
*/
#include <iostream>
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>  
#pragma comment(lib,"getopt.lib") 
#include "mxpcie.h"
using namespace std;

#define UNSET 	-1

enum action_type {
	GET_MODE = 0,
	SET_MODE = 1
};

struct action_struct {
	int type;
	int port;
	int number;
	int time;
};

void usage(FILE *fp)
{
	fprintf(fp, "Usage:\n");
	fprintf(fp, "	mx-pcie-ctl -i <PCIE Slot index>  [-m <PCIE power status>]\n");
	fprintf(fp, "OPTIONS:\n");
	fprintf(fp, "	-i <#PCIE Slot index>\n");
	fprintf(fp, "		Set target index: [0..]\n");
	fprintf(fp, "	-m <#pcie power status>\n");
	fprintf(fp, "		Set pcie slot power status\n");
	fprintf(fp, "		1 --> power on\n");
	fprintf(fp, "		0 --> power off\n");	
	fprintf(fp, "	-t <#PCIE Reset Slot Delay Time>\n");
	fprintf(fp, "		Set mini second [200...]\n");
	fprintf(fp, "\n");
	fprintf(fp, "Example:\n");
	fprintf(fp, "	Set PCIE slot 0 power on\n");
	fprintf(fp, "	# mx-pcie-ctl -i 0 -m 1\n");
	fprintf(fp, "	Set PCIE slot 0 power on with delay Time 200ms\n");
	fprintf(fp, "	# mx-pcie-ctl -i 0 -m 1 -t 200\n");
}

int my_atoi(const char *nptr, int *number)
{
	int tmp;

	tmp = atoi(nptr);
	if (tmp != 0) {
		*number = tmp;
		return 0;
	}
	else {
		if (!strcmp(nptr, "0")) {
			*number = 0;
			return 0;
		}
	}
	return -1;
}

void do_action(struct action_struct action)
{
	switch (action.type) {
	case GET_MODE:
		action.number = GetPCIESlotStatus(action.port);
		if (action.number < 0) {
			fprintf(stderr, "Failed to get PCIE power status\n");
			exit(1);
		}
		printf("PCIE slot %d power status: %d\n", action.port, action.number);
		break;
	case SET_MODE:
		if (action.time != 0)
		{
			if (SetPCIESlotStatusWithReset(action.port, action.number, action.time) < 0) {
				fprintf(stderr, "Failed to set PCIE power status\n");
				exit(1);
			}
		}
		else
		{
			if (SetPCIESlotStatus(action.port, action.number) < 0) {
				fprintf(stderr, "Failed to set PCIE power status\n");
				exit(1);
			}

		}
		printf("Set OK.\n");
		break;
	}
}

int main(int argc, char**argv)
{
	struct action_struct action = { GET_MODE, UNSET, UNSET };
	int c;

	while (1) {
		c = getopt(argc, argv, "hi:m:t:");
		if (c == -1)
			break;

		switch (c) {
		case 'h':
			usage(stdout);
			exit(0);
		case 'i':
			if (my_atoi(optarg, &action.port) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		case 'm':
			action.type = SET_MODE;
			if (my_atoi(optarg, &action.number) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		case 't':
			if (my_atoi(optarg, &action.time) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		default:
			printf("PCIE slot %d type: %d\n", action.port, action.type);
			usage(stderr);
			exit(99);
		}
	}

	if (optind < argc) {
		fprintf(stderr, "non-option arguments found: ");
		while (optind < argc)
			fprintf(stderr, "%s ", argv[optind++]);
		fprintf(stderr, "\n");

		usage(stderr);
		exit(99);
	}
	if (action.port == UNSET) {
		fprintf(stderr, "port number is unset\n");
		usage(stderr);
		exit(99);
	}
	do_action(action);
	exit(0);
}