/*
Copyright (C) MOXA Inc. All rights reserved.

This software is distributed under the terms of the
MOXA License.  See the file COPYING-MOXA for details.
*/
#include <iostream>
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>  
#pragma comment(lib,"getopt.lib") 
#include "mxpciereset.h"
using namespace std;

#define UNSET 	-1

enum action_type {
	GET_MODE = 0,
	SET_MODE = 1
};

struct action_struct {
	int type;
	int port;
	int time;
};

void usage(FILE *fp)
{
	fprintf(fp, "Usage:\n");
	fprintf(fp, "	mx-pciereset-ctl -i <PCIE Slot index> \n");
	fprintf(fp, "OPTIONS:\n");
	fprintf(fp, "	-i <#PCIE Reset Slot index>\n");
	fprintf(fp, "		Set target index: [0..]\n");
	fprintf(fp, "	-t <#PCIE Reset Slot Delay Time>\n");
	fprintf(fp, "		Set mini second [200...]\n");
	fprintf(fp, "\n");
	fprintf(fp, "Example:\n");
	fprintf(fp, "	Set PCIE Reset slot 0 \n");
	fprintf(fp, "	# mx-pciereset-ctl -i 0\n");
	fprintf(fp, "	Set PCIE Reset slot 0 delay Time 200ms\n");
	fprintf(fp, "	# mx-pciereset-ctl -i 0 -t 200\n");
}

int my_atoi(const char *nptr, int *number)
{
	int tmp;

	tmp = atoi(nptr);
	if (tmp != 0) {
		*number = tmp;
		return 0;
	}
	else {
		if (!strcmp(nptr, "0")) {
			*number = 0;
			return 0;
		}
	}
	return -1;
}

void do_action(struct action_struct action)
{
	int status = 0;
	switch (action.type) {
	case GET_MODE:
		status = GetRESETSlotStatus(action.port);
		if (status < 0) {
			fprintf(stderr, "Failed to get PCIE power status\n");
			exit(1);
		}
		printf("PCIE slot %d power status: %d\n", action.port, status);
		break;
	case SET_MODE:
		if (SetRESETSlotStatus(action.port,action.time) < 0) {
			fprintf(stderr, "Failed to Reset PCIE power status\n");
			exit(1);
		}
		printf("Set OK.\n");		
		break;
	}
}

int main(int argc, char**argv)
{
	struct action_struct action = { GET_MODE, UNSET, UNSET };
	int c;

	while (1) {
		c = getopt(argc, argv, "hi:t:");
		if (c == -1)
			break;

		switch (c) {
		case 'h':
			usage(stdout);
			exit(0);
		case 'i':
			action.type = SET_MODE;
			if (my_atoi(optarg, &action.port) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		case 't':
			action.type = SET_MODE;
			if (my_atoi(optarg, &action.time) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		
		default:
			printf("PCIE Reset slot %d type: %d\n", action.port, action.type);
			usage(stderr);
			exit(99);
		}
	}

	if (optind < argc) {
		fprintf(stderr, "non-option arguments found: ");
		while (optind < argc)
			fprintf(stderr, "%s ", argv[optind++]);
		fprintf(stderr, "\n");

		usage(stderr);
		exit(99);
	}
	if (action.port == UNSET) {
		fprintf(stderr, "port number is unset\n");
		usage(stderr);
		exit(99);
	}
	do_action(action);
	exit(0);
}