/*
Copyright (C) MOXA Inc. All rights reserved.

This software is distributed under the terms of the
MOXA License.  See the file COPYING-MOXA for details.
*/
#include <iostream>
#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <getopt.h>  
#pragma comment(lib,"getopt.lib") 
#include "mxsim.h"
using namespace std;

#define UNSET 	-1

enum action_type {
	GET_MODE = 0,
	SET_MODE = 1
};

struct action_struct {
	int type;
	int socket;
	int slot;
};

void usage(FILE *fp)
{
	fprintf(fp, "Usage:\n");
	fprintf(fp, "	mx-sim-ctl -s <Module socket index>  [-i <SIM slot>]\n");
	fprintf(fp, "OPTIONS:\n");
	fprintf(fp, "	-s <#Module socket index>\n");
	fprintf(fp, "		Target module socket index: [1..]\n");
	fprintf(fp, "	-i <#SIM slot>\n");
	fprintf(fp, "		Set SIM slot to\n");
	fprintf(fp, "		1 --> SIM slot 1\n");
	fprintf(fp, "		2 --> SIM slot 2\n");
	fprintf(fp, "\n");
	fprintf(fp, "Example:\n");
	fprintf(fp, "	Set SIM slot of module socket 1 to SIM slot 2\n");
	fprintf(fp, "	# mx-sim-ctl -s 1 -i 2\n");
}

int my_atoi(const char *nptr, int *number)
{
	int tmp;

	tmp = atoi(nptr);
	if (tmp != 0) {
		*number = tmp;
		return 0;
	}
	else {
		if (!strcmp(nptr, "0")) {
			*number = 0;
			return 0;
		}
	}
	return -1;
}

void do_action(struct action_struct action)
{
	switch (action.type) {
	case GET_MODE:
		action.slot = GetSIMSlot(action.socket - 1) + 1;
		if (action.slot < 0) {
			fprintf(stderr, "Failed to get SIM index\n");
			exit(1);
		}
		printf("Module socket %d SIM slot: %d\n", action.socket, action.slot);
		break;
	case SET_MODE:
		if (SetSIMSlot(action.socket - 1, action.slot - 1) < 0) {
			fprintf(stderr, "Failed to set SIM index\n");
			exit(1);
		}
		printf("Set OK.\n");		
		break;
	}
}

int main(int argc, char**argv)
{
	struct action_struct action = { GET_MODE, UNSET, UNSET };
	int c;

	while (1) {
		c = getopt(argc, argv, "hs:i:");
		if (c == -1)
			break;

		switch (c) {
		case 'h':
			usage(stdout);
			exit(0);
		case 's':
			if (my_atoi(optarg, &action.socket) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		case 'i':
			action.type = SET_MODE;
			if (my_atoi(optarg, &action.slot) != 0) {
				fprintf(stderr, "%s is not a number\n", optarg);
				exit(1);
			}
			break;
		default:
			printf("SIM %d type: %d\n", action.socket, action.type);
			usage(stderr);
			exit(99);
		}
	}

	if (optind < argc) {
		fprintf(stderr, "non-option arguments found: ");
		while (optind < argc)
			fprintf(stderr, "%s ", argv[optind++]);
		fprintf(stderr, "\n");

		usage(stderr);
		exit(99);
	}
	if (action.socket == UNSET) {
		fprintf(stderr, "port number is unset\n");
		usage(stderr);
		exit(99);
	}
	do_action(action);
	exit(0);
}