#!/bin/bash
#
# shellcheck disable=SC2317

BXPA101M_SYSPATH=("/sys/devices/pci0000:00/0000:00:14.0/usb1/1-4")
export BXPA101M_SYSPATH

SYSPATH=()

function get_model_name_from_dmi_type12() {
        /usr/sbin/dmidecode -t 12 |
                grep "Option " |
                awk -F ':' '{print substr($2,1,11)}' |
                sed 's/ //g'
}

function get_model_name_from_dmi_type1() {
        /usr/sbin/dmidecode -t 1 |
                grep "Product Name" |
                awk -F ':' '{print $2}' |
                sed 's/ //g'
}

function get_product_name() {
        local prodctu_name

        for name in $(get_model_name_from_dmi_type12); do
                prodctu_name="${name}"
                break
        done

        if [[ -z "${prodctu_name}" ]]; then
                for name in $(get_model_name_from_dmi_type1); do
                        prodctu_name="${name}"
                        break
                done
        fi

        if [[ -z "${prodctu_name}" ]]; then
                echo "Unsupported model"
                exit 38
        fi

        echo "${prodctu_name}"
}

function get_module_name() {
        local slot=$1
        local syspath=${SYSPATH[$slot]}

        local module_name=
        local module_sw_version=
        local manufacturer=

        module_name=$(mmcli -m "${syspath}" -K --timeout=30 | grep model | awk '{print $3}')
        module_sw_version=$(mmcli -m "${syspath}" -K --timeout=30 | grep modem.generic.revision | awk '{print $3}')
        manufacturer=$(mmcli -m "${syspath}" -K --timeout=30 | grep manufacturer | awk '{print $3}')
        if [ "${manufacturer}" = "Telit" ]; then
                if [[ ${module_name} =~ "LE910C"* ]]; then
                        echo "TELIT_LE910CX"
                        return 0
                fi
                if [[ ${module_name} =~ "FN990"* ]]; then
                        echo "TELIT_FN990"
                        return 0
                fi
        fi

        if [ "${manufacturer}" = "Quectel" ]; then
                if [[ ${module_name} =~ "RM520N"* ]]; then
                        echo "QUECTEL_RM520"
                        return 0
                fi
        fi

        if [ "${module_name}" = "QUECTEL" ]; then
                if [[ ${module_sw_version} =~ "EC20"* ]]; then
                        echo "EC20"
                        return 0
                fi
        fi

        return 0
}

function telit_quectel_lte_power_on() {
        local slot=$1

        mx-module-ctl -s "${slot}" -d high #w_disable pin
        sleep 1

        mx-module-ctl -s "${slot}" -r low #reset pin
        sleep 1

        mx-module-ctl -s "${slot}" -p high #power pin

        return 0
}

function telit_le910cx_power_off() {
        local slot=$1
        local syspath=${SYSPATH[$slot]}
        local response

        echo "pull down w_disable"
        mx-module-ctl -s "${slot}" -d low #w_disable pin

        sleep 0.5

        while true; do
                response=$(mmcli -m "${syspath}" --timeout=30 --command="AT#SHDN")
                if [ "${response}" = "response: ''" ]; then
                        break
                fi
        done

        echo "monitor alive pin"
        for _ in {1..40}; do
                if mx-module-ctl -s "${slot}" --mod | grep -q "inactive"; then
                        echo "the module is deactivated !"
                        break
                fi
                sleep 0.5
        done

        echo "turn off the power"
        mx-module-ctl -s "${slot}" -p low #power pin

        return 0
}

function power_on() {
        local slot=$1

        telit_quectel_lte_power_on "${slot}"

        return 0
}

function power_off() {
        local slot=$1
        local module_name=

        module_name=$(get_module_name "${slot}")
        if [ "${module_name}" == "TELIT_LE910CX" ]; then
                telit_le910cx_power_off "${slot}"
                return 0
        fi

        echo "No Know Module"
        return 1
}

function init() {
        local product_name
        product_name=$(get_product_name)

        if [ -z "${product_name}" ]; then
                echo "No Support the product"
                exit 1
        fi
        eval 'SYSPATH=("${'"${product_name}"'_SYSPATH[@]}")'
}

function usage() {
        echo "Usage: $0 -o|-f [-s 0|1|2 (default 0)]"
        echo "  -o: power on"
        echo "  -f: power off"
        echo "  -s: slot to use for cellular"
}

function main() {
        local slot=0
        local power_ops=""

        init
        while getopts "hs:of" opt; do
                case "${opt}" in
                s)
                        slot="${OPTARG}"
                        ;;
                o)
                        power_ops="power_on"
                        ;;
                f)
                        power_ops="power_off"
                        ;;
                h)
                        usage
                        return 1
                        ;;
                *)
                        usage
                        return 1
                        ;;
                esac
        done
        shift $((OPTIND - 1))

        if [ -z "$power_ops" ]; then
                usage
                return 1
        fi

        $power_ops "$slot"
        return "$?"
}

main "$@"
