/**
 * @file mcm-network-info.h
 *
 * This library is for Moxa Connection Management
 *
 * @author York LY Chang <YorkLY.Chang@moxa.com>
 */

#ifndef _MCM_NETWORK_INFO_H_
#define _MCM_NETWORK_INFO_H_
#include "mcm-base-info.h"

/**
 * McmConnectionState:
 *
 * Enumeration of possible connection states.
 */
typedef enum {
    CON_STATE_ERROR              = -1, /**< Fatal error */
    CON_STATE_INITIAL            = 0,  /**< The connection is currently being initialized */
    CON_STATE_DEVICE_UNAVAILABLE = 1,  /**< The network interface unavailable to use */
    CON_STATE_DEVICE_READY       = 2,  /**< The network interface ready for next action */
    CON_STATE_DISABLING          = 3,  /**< The network interface is currently transitioning to the ::CON_STATE_DISABLED state.*/
    CON_STATE_DISABLED           = 4,  /**< The network interface is not enabled and is down. */
    CON_STATE_CONFIGURED         = 5,  /**< The network interface applied connection setting from profile. */
    CON_STATE_CONFIGURE_FAILED   = 6,  /**< setting connection failed, probably lower layer failure or wrong setting */
    CON_STATE_DISCONNECTED       = 7,  /**< The network interface connection configured but data connection disconnect. */
    CON_STATE_RECONNECT          = 8,  /**< Under data recovery process */
    CON_STATE_CONNECTED          = 9,  /**< The network interface connection configured and data connection connected. */
} McmConnectionState;

/**
 * @brief A help function for get network informations.
 *
 * @param interfaceName (input) Interface name.
 *
 * @param network_info (output) A network infomations structure.
 *
 * @return If get ethernet properties success return 0. If fail, return error number.
 */
int mcm_get_network_info(char *interfaceName, mcm_network_info *network_info);

/**
 * @brief A help function for get network status.
 *
 * @param interfaceName (input) Interface name.
 *
 * @param network_status (output) A network status structure.
 *
 * @return If get ethernet properties success return 0. If fail, return error number.
 */
int mcm_get_network_status(char *interfaceName, mcm_network_status *network_status);

/**
 * @brief A help function for get network profiles.
 *
 * @param interfaceName (input) Interface name.
 *
 * @param profiles (output) A network profiles structure.
 *
 * @param num (output) network profiles numbers.
 *
 * @return If get ethernet properties success return 0. If fail, return error number.
 */
int mcm_get_network_profiles(char *interfaceName, int *num, mcm_profiles **profiles);

/**
 * @brief A help function for get network property.
 *
 * @param interfaceName (input) Interface name.
 *
 * @param propertyName (input) property name.
 *
 * @param propertyValue (output) property value.
 *
 * @return If get ethernet properties success return 0. If fail, return error number.
 */
int mcm_network_get_property(char *interfaceName, char *propertyName, char **propertyValue);

#endif
