#!/bin/bash
#
# shellcheck disable=SC2317

V3000_SYSPATH=("/sys/devices/pci0000:00/0000:00:14.0/usb1/1-6"
    "/sys/devices/pci0000:00/0000:00:14.0/usb2/2-3"
    "/sys/devices/pci0000:00/0000:00:14.0/usb2/2-4")
export V3000_SYSPATH

SYSPATH=()

function get_product_name() {
    local prodctu_name
    prodctu_name=$(/usr/sbin/dmidecode -t 1 | grep 'Product Name' | cut -d ':' -f 2 | tr -d ' ' | tr -d '\n')
    echo "${prodctu_name}"
}

function get_module_hw_id() {
    local hw_id=15 #LTE module
    local lte_id=15
    local value
    local slot=$1
    local response=

    response=$(mx-module-ctl -s "${slot}" --mod)
    value=$(echo "${response}" | grep 'CONFIG_0' | cut -d '=' -f 2)
    if [ -z "${value}" ]; then
        echo "${lte_id}"
        return
    fi
    hw_id=$((value * 1))

    value=$(echo "${response}" | grep 'CONFIG_1' | cut -d '=' -f 2)
    if [ -z "${value}" ]; then
        echo "${lte_id}"
        return
    fi
    hw_id=$((hw_id + value * 2))

    value=$(echo "${response}" | grep 'CONFIG_2' | cut -d '=' -f 2)
    if [ -z "${value}" ]; then
        echo "${lte_id}"
        return
    fi
    hw_id=$((hw_id + value * 4))

    value=$(echo "${response}" | grep 'CONFIG_3' | cut -d '=' -f 2)
    if [ -z "${value}" ]; then
        echo "${lte_id}"
        return
    fi
    hw_id=$((hw_id + value * 8))

    echo "${hw_id}"
}

function get_module_name() {
    local slot=$1
    local syspath=${SYSPATH[$slot]}

    local module_name=
    local module_sw_version=
    local manufacturer=

    module_name=$(mmcli -m "${syspath}" -K | grep model | awk '{print $3}')
    module_sw_version=$(mmcli -m "${syspath}" -K | grep modem.generic.revision | awk '{print $3}')
    manufacturer=$(mmcli -m "${syspath}" -K | grep manufacturer | awk '{print $3}')

    if [ "${manufacturer}" = "Telit" ]; then
        if [[ ${module_name} =~ "LE910C"* ]]; then
            echo "TELIT_LE910CX"
            return 0
        fi
        if [[ ${module_name} =~ "FN990"* ]]; then
            echo "TELIT_FN990"
            return 0
        fi
    fi

    if [ "${manufacturer}" = "Quectel" ]; then
        if [[ ${module_name} =~ "RM520N"* ]]; then
            echo "QUECTEL_RM520"
            return 0
        fi
    fi

    if [ "${module_name}" = "QUECTEL" ]; then
        if [[ ${module_sw_version} =~ "EC20"* ]]; then
            echo "EC20"
            return 0
        fi
    fi

    return 0
}

function telit_quectel_lte_power_on() {
    local slot=$1

    mx-module-ctl -s "${slot}" -d high #w_disable pin
    sleep 1

    mx-module-ctl -s "${slot}" -r high #reset pin
    sleep 1

    mx-module-ctl -s "${slot}" -p high #power pin

    return 0
}

function quectel_ec20_power_off() {
    local slot=$1

    mx-module-ctl -s "${slot}" -r high #reset pin
    sleep 1

    mx-module-ctl -s "${slot}" -p high #power pin
    return 0
}

function telit_le910cx_power_off() {
    local slot=$1
    local syspath=${SYSPATH[$slot]}
    local response

    echo "pull down w_disable"
    mx-module-ctl -s "${slot}" -d low #w_disable pin

    while true; do
        response=$(mmcli -m "${syspath}" --command="AT#SHDN")
        if [ "${response}" = "response: ''" ]; then
            break
        fi
    done

    echo "monitor alive pin"
    for _ in {1..40}; do
        state=$(mx-module-ctl -s "${slot}" --mod | grep LE910C4-WWXD | cut -d '=' -f 2)
        if [ "$state" == "0" ]; then
            echo "the module is deactivated !"
            break
        fi
        sleep 0.5
    done

    echo "turn off the power"
    mx-module-ctl -s "${slot}" -p low #power pin

    return 0
}

function telit_fn990_5g_power_on() {
    local slot=$1

    echo "turn on the power"

    mx-module-ctl -s "${slot}" -p high
    sleep 1

    mx-module-ctl -s "${slot}" -t high
    sleep 1

    mx-module-ctl -s "${slot}" -r high

    return 0
}

function telit_fn990_5g_power_off() {
    local slot=$1

    echo "turn off the power (FC_PWROFF)"
    mx-module-ctl -s "${slot}" -t low

    echo "monitor alive pin"
    for _ in {1..40}; do
        state=$(mx-module-ctl -s "${slot}" --mod | grep FN990 | cut -d '=' -f 2)
        if [ "$state" == "0" ]; then
            echo "the module is deactivated !"
            break
        fi
        sleep 0.5
    done

    echo "turn off the power"

    mx-module-ctl -s "${slot}" -p low

    return 0
}

function quectel_rm520_5g_power_on() {
    local slot=$1

    echo "turn on the power"
    mx-module-ctl -s "${slot}" -r high
    sleep 1

    mx-module-ctl -s "${slot}" -p high
    sleep 1

    mx-module-ctl -s "${slot}" -t high

    return 0
}

function quectel_rm520_5g_power_off() {
    local response
    local slot=$1
    local syspath=${SYSPATH[$slot]}

    while true; do
        response=$(mmcli -m "${syspath}" --command="AT+CFUN=0,0")
        if [ "${response}" = "response: ''" ]; then
            break
        fi
    done

    sleep 0.1

    echo "set reset pin low"
    mx-module-ctl -s "${slot}" -r low

    sleep 0.1

    echo "set FC power low"
    mx-module-ctl -s "${slot}" -t low
    sleep 1

    echo "turn off the power"
    mx-module-ctl -s "${slot}" -p low

    return 0
}

function power_on() {
    local module_hw_id
    local slot=$1

    module_hw_id=$(get_module_hw_id "${slot}")

    if [ "${module_hw_id}" = "15" ]; then #Telit LE910Cx or Quectel EC20
        telit_quectel_lte_power_on "${slot}"
    elif [ "${module_hw_id}" = "12" ]; then
        telit_fn990_5g_power_on "${slot}"
    elif [ "${module_hw_id}" = "13" ]; then
        quectel_rm520_5g_power_on "${slot}"
    fi
    return 0
}

function power_off() {
    local slot=$1
    local module_name=

    module_name=$(get_module_name "${slot}")

    if [ "${module_name}" = "EC20" ]; then
        quectel_ec20_power_off "${slot}"
        return 0
    fi

    if [ "${module_name}" = "TELIT_LE910CX" ]; then
        telit_le910cx_power_off "${slot}"
        return 0
    fi

    if [ "${module_name}" = "TELIT_FN990" ]; then
        telit_fn990_5g_power_off "${slot}"
        return 0
    fi

    if [ "${module_name}" = "QUECTEL_RM520" ]; then
        quectel_rm520_5g_power_off "${slot}"
        return 0
    fi

    echo "No Know Module"
    exit 1
}

function init() {
    local product_name
    product_name=$(get_product_name)

    if [ -z "${product_name}" ]; then
        echo "No Support the product"
        exit 1
    fi
    eval 'SYSPATH=("${'"${product_name}"'_SYSPATH[@]}")'
}

function usage() {
    echo "Usage: $0 -o|-f [-s 0|1|2 (default 0)]"
    echo "  -o: power on"
    echo "  -f: power off"
    echo "  -s: slot to use for cellular"
}

function main() {
    local slot=0
    local power_ops=""

    init
    while getopts "hs:of" opt; do
        case "${opt}" in
        s)
            slot="${OPTARG}"
            ;;
        o)
            power_ops="power_on"
            ;;
        f)
            power_ops="power_off"
            ;;
        h)
            usage
            return 1
            ;;
        *)
            usage
            return 1
            ;;
        esac
    done
    shift $((OPTIND - 1))

    if [ -z "$power_ops" ]; then
        usage
        return 1
    fi

    $power_ops "$slot"
    return "$?"
}

main "$@"
exit "$?"
