#!/bin/bash
#
# Copyright (C) 2023 MOXA Inc. All rights reserved.
# This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
# See the file LICENSE for details.
#
# Authors:
#       2023  Wilson YS Huang  <wilsonys.huang@moxa.com>

# shellcheck disable=SC2317 # Don't warn about unreachable commands in this file

source "${INSTL_SCRIPT_DIR}/lib/util.sh"
source "${INSTL_SCRIPT_DIR}/lib/logger.sh"
log::init "${INSTL_LOG_FILE}" "${INSTL_LOG_LEVEL}"

set -e

process_file_uninstall() {
        local repo_name="$1"
        local repo_tag="$2"
        local repo_ver

        if util::is_exists "${INSTL_SRC_VERSION_FILE}" && grep -wq "${repo_name}" "${INSTL_SRC_VERSION_FILE}"; then
                repo_ver=$(grep -w "${repo_name}" "${INSTL_SRC_VERSION_FILE}" | awk -F ':' '{print $2}')
        else
                repo_ver="unknown"
        fi

        if util::confirm_prompt "Do you want to uninstall ${repo_name} (${repo_ver})?" "${INSTL_PROMPT_AUTO_YES}"; then
                source "${INSTL_SCRIPT_DIR}/uninstall.d/${repo_name}.uninstall.sh" "${repo_tag}" "${repo_ver}"

                sed -i "/$repo_name/d" "${INSTL_SRC_VERSION_FILE}"
                return 0
        else
                log::info "Skip uninstall ${repo_name}"
                return 1
        fi
}

uninstall_debian_based_dev_tools() {
        apt-get remove build-essential linux-headers-"$(uname -r)" -y
}

uninstall_redhat_based_dev_tools() {
        yum groupremove "Development Tools" -y
        yum remove kernel-devel-"$(uname -r)" kernel-headers-"$(uname -r)" -y
}

main() {
        local drivers_list
        local tools_list
        local driver_available
        local tool_available

        driver_available=true
        tool_available=true

        export PATH=$PATH:/usr/sbin

        if util::is_exists "${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/prerm.sh"; then
                log::info "=== Run pre-uninstall"
                source "${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/prerm.sh"
        fi

        log::info "=== Uninstall driver"
        drivers_list="${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/drivers"
        if ! util::parse_product_file "${drivers_list}" "${LINUX_KERNEL_MAJOR_VER}" "${LINUX_KERNEL_MINOR_VER}" process_file_uninstall; then
                driver_available=false
        fi

        log::info "=== Uninstall tool"
        tools_list="${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/tools"
        if ! util::parse_product_file "${tools_list}" "${LINUX_KERNEL_MAJOR_VER}" "${LINUX_KERNEL_MINOR_VER}" process_file_uninstall; then
                tool_available=false
        fi

        if [[ $driver_available = false ]] && [[ $tool_available = false ]]; then
                exit 3
        else
                if util::is_exists "${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/postrm.sh"; then
                        log::info "=== Run post-uninstall"
                        source "${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/postrm.sh"
                fi
                exit 0
        fi
}

main
