/*
 * Copyright (C) MOXA Inc. All rights reserved.
 * Authors:
 *     2024  Wilson YS Huang  <wilsonys.huang@moxa.com>
 * This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
 * See the file LICENSE for details.
 */

#define _GNU_SOURCE
#include <logger.h>
#include <mcu.h>
#include <tty.h>

#define APP_WDT_TICK_INTERVAL 4000
static pthread_mutex_t mcu_rw_lock = PTHREAD_MUTEX_INITIALIZER;
static int32_t         mcu_fd      = -1;

static void *app_watchdog_tick_thread(void *arg)
{
    bool         *ticking      = (bool *)arg;
    mcu_packet    send_mcu_pkt = {0};
    mcu_packet   *recv_mcu_pkt = NULL;
    ssize_t       pkt_sz       = 0;
    struct pollfd pollfd[1]    = {0};
    int32_t       rc           = -1;

    pkt_sz           = make_mcu_packet(&send_mcu_pkt, (uint8_t *)MX_MCU_APP_WDT_TIMEOUT_CMD, NULL, 0);
    pollfd[0].fd     = mcu_fd;
    pollfd[0].events = POLLOUT;
    recv_mcu_pkt     = (mcu_packet *)calloc(1, sizeof(mcu_packet) + MAX_DATA_SZ + sizeof(uint8_t));
    if (recv_mcu_pkt == NULL) {
        log_error("Allocate receive packet failed.");
        return NULL;
    }

    log_info("Start app watchdog tick, interval: %d seconds", APP_WDT_TICK_INTERVAL);

    while (*ticking) {
        rc = poll(pollfd, COUNT_OF_POLLFD(pollfd), 1000);
        if (rc < 0) {
            log_error("App watchdog ticking failed");
            break;
        }

        if (pollfd[0].revents & POLLOUT) {
            log_debug("App watchdog ticking...");
            mcu_write(&send_mcu_pkt, pkt_sz);
            sleep_ms(APP_WDT_TICK_INTERVAL);
            mcu_read(recv_mcu_pkt, MAX_DATA_SZ);
        }
    }

    free(recv_mcu_pkt);
    log_info("Stop app watchdog tick");
    return NULL;
}

static bool app_watchdog_callback(void *arg)
{
    static pthread_t tid;
    static bool      ticking;
    uint8_t         *data = (uint8_t *)arg;

    log_debug("tid: %ld", tid);

    if (data[0] == 0) {
        ticking = false;
        pthread_join(tid, NULL);
        tid = 0;
    }
    else if (data[0] == 1) {
        ticking = true;
        if (tid == 0) {
            if (pthread_create(&tid, NULL, app_watchdog_tick_thread, &ticking) != 0) {
                log_warn("Failed to create app watchdog tick thread");
                return false;
            }
            else {
                log_debug("Create app watchdog tick thread");
                pthread_setname_np(tid, "app watchdog tick");
            }
        }
    }
    else {
        log_warn("Unknown app watchdog mode");
        return false;
    }

    return true;
}
struct {
    char   *command;
    uint8_t feature_bit;
    bool (*command_callback_fn)(void *arg);
} mcu_commands_tbl[] = {
    {MX_MCU_VERSION_CMD, FEAT_MCU_VERSION, NULL},
    {MX_MCU_RELAY_MODE_CMD, FEAT_MCU_RELAY_MODE, NULL},
    {MX_MCU_WDT_RESET_MODE_CMD, FEAT_MCU_WDT_RESET_MODE, NULL},
    {MX_MCU_WDT_RELAY_MODE_CMD, FEAT_MCU_WDT_RELAY_MODE, NULL},
    {MX_MCU_POWEROFF_RELAY_MODE_CMD, FEAT_MCU_POWEROFF_RELAY_MODE, NULL},
    {MX_MCU_APP_WDT_RELAY_MODE_CMD, FEAT_MCU_APP_WDT_RELAY_MODE, NULL},
    {MX_MCU_APP_WDT_RESET_MODE_CMD, FEAT_MCU_APP_WDT_RESET_MODE_CMD, NULL},
    {MX_MCU_APP_WDT_TIMEOUT_CMD, FEAT_MCU_APP_WDT_TIMEOUT_CMD, app_watchdog_callback},
    {NULL, -1, NULL}};

static uint8_t calculate_checksum(const uint8_t *data, size_t length)
{
    uint8_t checksum = 0;

    for (uint8_t i = 0; i < length; ++i) {
        checksum += data[i];
    }

    return checksum;
}

int32_t mcu_device_open(const char *tty_name)
{
    int32_t fd = 0;

    fd = open_tty(tty_name);
    if (flock(fd, LOCK_EX | LOCK_NB) < 0) {
        log_error("Error: MCU device is locked.\n");
        return -1;
    }

    mcu_fd = fd;

    return fd;
}

ssize_t mcu_read(void *buf, size_t nbytes)
{
    ssize_t rc;
    pthread_mutex_lock(&mcu_rw_lock);
    rc = read(mcu_fd, buf, nbytes);
    pthread_mutex_unlock(&mcu_rw_lock);

    return rc;
}

ssize_t mcu_write(void *buf, size_t nbytes)
{
    ssize_t rc;
    pthread_mutex_lock(&mcu_rw_lock);
    rc = write(mcu_fd, buf, nbytes);
    tcdrain(mcu_fd);
    pthread_mutex_unlock(&mcu_rw_lock);

    return rc;
}

bool is_mcu_command_support(uint8_t *command, int8_t features)
{
    if (command == NULL) {
        return false;
    }

    for (uint8_t i = 0; mcu_commands_tbl[i].command != NULL; ++i) {
        log_debug("Check MCU command %s", mcu_commands_tbl[i].command);
        if (!memcmp(command, mcu_commands_tbl[i].command, MX_MCU_CMD_SIZE) &&
            features & mcu_commands_tbl[i].feature_bit) {
            return true;
        }
    }

    return false;
}

bool invoke_mcu_command_callback_fn(uint8_t *command, void *data)
{
    if (command == NULL) {
        return false;
    }

    for (uint8_t i = 0; mcu_commands_tbl[i].command != NULL; ++i) {
        log_debug("Check MCU command %s", mcu_commands_tbl[i].command);
        if (!memcmp(command, mcu_commands_tbl[i].command, MX_MCU_CMD_SIZE) &&
            mcu_commands_tbl[i].command_callback_fn != NULL) {
            return mcu_commands_tbl[i].command_callback_fn(data);
        }
    }

    return false;
}

size_t make_mcu_packet(mcu_packet *mcu_pkt, uint8_t command[3], uint8_t *data, size_t data_sz)
{
    mcu_packet *pkt    = NULL;
    size_t      pkt_sz = 0;

    if (data_sz != 0) {
        pkt = (mcu_packet *)calloc(1, sizeof(mcu_packet) + data_sz + sizeof(uint8_t));
        memcpy(pkt->data, data, data_sz);
        pkt->data[data_sz] = 0xff - (calculate_checksum(data, data_sz) & 0xff);   // data checksum
        pkt_sz             = sizeof(mcu_packet) + data_sz + sizeof(uint8_t);
    }
    else {
        pkt          = (mcu_packet *)calloc(1, sizeof(mcu_packet) + 2 * sizeof(uint8_t));
        pkt->data[0] = 0x00;
        pkt->data[1] = 0xFF;
        pkt_sz       = sizeof(mcu_packet) + 2 * sizeof(uint8_t);
    }

    pkt->attn = MX_MCU_CMD;
    pkt->addr = MX_MCU_ADDR_BROADCAST;
    memcpy(pkt->command, command, MX_MCU_CMD_SIZE);
    pkt->data_sz = data_sz;
    pkt->hdr_chk = 0xff - ((pkt->attn + pkt->addr + pkt->command[0] + pkt->command[1] + pkt->command[2] + data_sz) & 0xff);

    memcpy(mcu_pkt, pkt, pkt_sz);

    free(pkt);

    return pkt_sz;
}

bool validate_mcu_packet(mcu_packet *pkt)
{
    uint8_t hdr_chk;
    uint8_t data_chk;

    hdr_chk = 0xff - ((pkt->attn + pkt->addr + pkt->command[0] + pkt->command[1] + pkt->command[2] + pkt->data_sz) & 0xff);
    if (hdr_chk != pkt->hdr_chk) {
        return false;
    }

    if (pkt->data_sz != 0) {
        data_chk = 0xff - (calculate_checksum(pkt->data, pkt->data_sz) & 0xff);
        if (data_chk != pkt->data[pkt->data_sz]) {
            return false;
        }
    }

    return true;
}
