#!/bin/bash -e
#
# Copyright (C) 2024 MOXA Inc. All rights reserved.
# This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
# See the file LICENSE for details.
#

set -e

source "${INSTL_SCRIPT_DIR}/lib/util.sh"
source "${INSTL_SCRIPT_DIR}/lib/logger.sh"
source "${INSTL_PRODUCT_DIR}/${PRODUCT_NAME}/config"
log::init "${INSTL_LOG_FILE}" "${INSTL_LOG_LEVEL}"

main() {
        # auto probe gpio-pca953x when boot up
        echo "gpio-pca953x" >/usr/lib/modules-load.d/gpio-pca953x.conf

        log::info "Install hardware sensor monitor lm-sensors packages and configuration file"
        {
                case "${OS_PACKAGE_MANAGER}" in
                "apt") apt-get install -y "lm-sensors" ;;
                "yum") yum install -y "lm_sensors" ;;
                esac
        } >>"${INSTL_LOG_FILE}"

        cat <<EOF >/lib/modules-load.d/it87.conf
it87
EOF
        if ! util::is_exists "/etc/sensors.d/"; then
                mkdir -p "/etc/sensors.d/"
        fi

        if ! util::is_exists "/etc/sensors.d/it8786.conf"; then
                adc_lsb_coeff="0.9"
                V3_3_Rin="6.19"
                V3_3_Rf="10"
                V5_Rin="10"
                V5_Rf="10"

                cat <<EOF >/etc/sensors.d/it8786.conf
chip "it8786-*"
    ignore in0
    ignore in1

    label in2 "V3.3"
    compute in2 @*$adc_lsb_coeff*(1+($V3_3_Rin/$V3_3_Rf)), @/($adc_lsb_coeff*(1+($V3_3_Rin/$V3_3_Rf)))
    set in2_min 3.3*0.9
    set in2_max 3.3*1.1

    label in3 "V5.0"
    compute in3 @*$adc_lsb_coeff*(1+($V5_Rin/$V5_Rf)), @/($adc_lsb_coeff*(1+($V5_Rin/$V5_Rf)))
    set in3_min 5.0*0.9
    set in3_max 5.0*1.1

    ignore in4
    ignore in5
    ignore in6
    ignore in7
    # VBat
    ignore in8

    ignore fan1
    ignore fan2
    ignore fan3

    label temp1 "SYSTEM"
    label temp2 "SYSTEM2"

    ignore temp3
    ignore intrusion0
EOF

        fi

        log::info "Copy and enable service/udev files on V3400"
        {
                pushd "${INSTL_TOP_DIR}/product.d/V3400" >/dev/null
                log::info "Rename serial device with ttyMX"
                cp 99-moxa-x86-serial-naming.rules /etc/udev/rules.d/
                log::info "Set LTE mPCIe SIM card slot as default slot 1"
                cp 00-moxa-v3400-set-sim-default.rules /etc/udev/rules.d/
                log::info "Set SSD disk hotswap auto mount"
                cp 96-moxa-disk.rules /etc/udev/rules.d/
                cp 95-sata-hotplug.rules /etc/udev/rules.d/
                udevadm control --reload-rules
                udevadm trigger

                log::info "Copy rtc battery detect to systemd service w/o enable"
                cp mx_rtc_battery_detect.service /lib/systemd/system/

                log::info "Copy SSD disk hotswap to systemd service w/o enable due to security concern"
                cp moxa-disk-hotswapd.service /lib/systemd/system/
                systemctl daemon-reload

                popd
        } >>"${INSTL_LOG_FILE}" 2>&1

        if uname -a | grep -q Debian; then
                log::info "Install firmware-atheros files for Wi-Fi modules on V3400"
                {
                        if ! dpkg -l | grep -q firmware-atheros; then
                                apt-get -y install firmware-atheros
                                ln -s /lib/firmware/ath10k/QCA6174/hw3.0/board.bin /lib/firmware/ath10k/pre-cal-pci-0000:15:00.0.bin
                                ln -s /lib/firmware/ath10k/QCA6174/board-2.bin /lib/firmware/ath10k/cal-pci-0000:15:00.0.bin
                                ln -s /lib/firmware/ath10k/QCA6174/hw3.0/firmware-4.bin /lib/firmware/ath10k/QCA6174/hw3.0/firmware-5.bin
                        fi
                } >>"${INSTL_LOG_FILE}" 2>&1
        fi

        log::info "Install Moxa Connection Manager (MCM) packages"
        if util::confirm_prompt "Do you wish to install Moxa Connection Manager (${CFG_MCM_VER})?" "${INSTL_PROMPT_AUTO_YES}"; then
                echo "Start to install Moxa Connection Manager, beware it will set LAN1 as DHCP by MCM..."
                install_mcm
                return 0
        else
                log::info "Skip install Moxa Connection Manager (MCM) packages"
                return 1
        fi
}

install_mcm() {
        local os_codename
        local os_ver
        local pkg_target

        os_codename="$(awk -v opt="VERSION_CODENAME" -F= '$1==opt { print $2 ;}' /etc/os-release | tr -d '"')"
        os_ver="$(awk -v opt="VERSION_ID" -F= '$1==opt { print $2 ;}' /etc/os-release | tr -d '"')"
        pkg_target="mcm"

        if util::is_exists "${INSTL_TOP_DIR}"/"${pkg_target}"/"${os_codename}_${os_ver}/${CFG_MCM_VER}"; then
                case "${OS_PACKAGE_MANAGER}" in
                "apt")
                        apt-get -y install "${INSTL_TOP_DIR}"/"${pkg_target}"/"${os_codename}"_"${os_ver}"/"${CFG_MCM_VER}"/*.deb >>"${INSTL_LOG_FILE}"
                        ;;
                esac
        else
                echo "Moxa Connection Manager ${os_codename}_${os_ver}/${CFG_MCM_VER} is not exist"
                exit 0
        fi
}

main
