#!/bin/bash
#
# Copyright (C) 2023 MOXA Inc. All rights reserved.
# This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
# See the file LICENSE for details.
#
# Authors:
#       2023  Wilson YS Huang  <wilsonys.huang@moxa.com>

source "${INSTL_SCRIPT_DIR}/lib/util.sh"
source "${INSTL_SCRIPT_DIR}/lib/logger.sh"
source "${INSTL_SCRIPT_DIR}/lib/driver.sh"
log::init "${INSTL_LOG_FILE}" "${INSTL_LOG_LEVEL}"

set -e

REPO_NAME="moxa-it87-wdt-driver"
SRC_PATH_PREFIX="${INSTL_TOP_DIR}/src/${REPO_NAME}"

main() {
        local src_tag
        local src_ver
        local drv_path
        local wdt_blacklist
        src_tag="$1"
        src_ver="$2"

        log::info "Installing ${REPO_NAME} (${src_ver})"

        if [[ "${LINUX_DISTRO_ID}" == "rhel" ]] || [[ "${LINUX_DISTRO_ID}" == "centos" ]]; then
                if drv::is_loaded "iTCO_wdt"; then
                        modprobe -r "iTCO_wdt"
                fi

                if drv::is_loaded "mei_wdt"; then
                        modprobe -r "mei_wdt"
                fi
        fi

        {
                pushd "${SRC_PATH_PREFIX}-${src_tag}" >/dev/null

                make clean
                make install

                # Compress module if kernel config compress type is enabled
                drv_path=$(modinfo -n it87_wdt 2>/dev/null || true)
                drv::compress_module "$drv_path"

                make load

                popd
        } >>"${INSTL_LOG_FILE}" 2>&1

        cat <<EOF >/lib/modules-load.d/it87_wdt.conf
it87_wdt
EOF

        cat <<EOF >/lib/modprobe.d/it87_wdt-option.conf
# timeout:Watchdog timeout in seconds, default=60 (int)
# nowayout:Watchdog cannot be stopped once started, default=0 (bool)
options it87_wdt nowayout=1 timeout=$CFG_IT87_WDT_TIMEOUT ldn_reset=1
EOF

        # blacklist non super IO wdt from grub
        {
                if [[ "${LINUX_DISTRO_ID}" == "rhel" ]] || [[ "${LINUX_DISTRO_ID}" == "centos" ]]; then
                        local ignore_wdt_drv
                        local ignored_list

                        ignore_wdt_drv="iTCO_wdt iTCO_vendor_support mei_wdt wdat_wdt"
                        ignored_list=$(drv::add_blacklist "$ignore_wdt_drv" "blacklist-wdt.conf")

                        if ! util::is_empty_str "$ignored_list"; then
                                cat <<EOF >/etc/dracut.conf.d/omit-wdt.conf
omit_dracutmodules+=" $ignore_wdt_drv "
omit_drivers+=" $ignore_wdt_drv "
EOF
                                dracut -c /etc/dracut.conf.d/omit-wdt.conf -f /boot/initramfs-"$(uname -r)".img "$(uname -r)"

                                for mod in ${ignored_list}; do
                                        grubby --args "$mod.blacklist=1 rd.driver.blacklist=$mod" --update-kernel ALL
                                        sed -i "/^KDUMP_COMMANDLINE_APPEND=/s/\"$/ rd.driver.blacklist=$mod\"/" /etc/sysconfig/kdump
                                done

                                # If kernel does not reserve memory to kdump, it would encounter error.
                                if kdumpctl status; then
                                        kdumpctl restart
                                        mkdumprd -f /boot/initramfs-"$(uname -r)"kdump.img
                                fi
                        fi
                else
                        wdt_blacklist="module_blacklist=iTCO_wdt,mei_wdt,iTCO_vendor_support,mei_wdt,wdat_wdt"
                        if ! grep -wq "${wdt_blacklist}" "/etc/default/grub"; then
                                sed -i "s/GRUB_CMDLINE_LINUX=\"[^\"]*/& ${wdt_blacklist}/" /etc/default/grub
                                update-grub2
                        fi

                        echo -e "blacklist iTCO_wdt\nblacklist iTCO_vendor_support" >/lib/modprobe.d/iTCO-blacklist.conf
                fi
        } >>"${INSTL_LOG_FILE}" 2>&1

        # Install watchdog daemon
        {
                if [[ "${OS_PACKAGE_MANAGER}" == "apt" ]]; then
                        apt-get update
                        apt-get install watchdog -y
                elif [[ "${OS_PACKAGE_MANAGER}" == "yum" ]] && [[ "${LINUX_DISTRO_ID}" == "rhel" ]] || [[ "${LINUX_DISTRO_ID}" == "rocky" ]]; then
                        yum makecache
                        yum install watchdog -y
                fi
        } >>"${INSTL_LOG_FILE}" 2>&1

        if [[ "${LINUX_DISTRO_ID}" != "centos" ]]; then
                log::info " Setup Watchdog Service"
                {
                        sed -i "/^#watchdog-device/s/^#//g" /etc/watchdog.conf
                        systemctl enable watchdog
                } >>"${INSTL_LOG_FILE}" 2>&1
        fi

        cat <<EOF >/etc/default/watchdog
# Start watchdog at boot time? 0 or 1
run_watchdog=1
# Start wd_keepalive after stopping watchdog? 0 or 1
run_wd_keepalive=0
# Load module before starting watchdog
watchdog_module="it87_wdt"
# Specify additional watchdog options here (see manpage).
EOF
}

main "$@"
