#!/usr/bin/env bash

set -e

if [ "$MODE" = "debug" ]; then
  set -x
fi

func_docker_openssl() {
  docker run -i --rm \
    -v /dev/urandom:/dev/random \
    -v /opt/moxa/dlm:/opt/moxa/dlm:rw \
    -v "${USER_DATA_DIR}:${USER_DATA_DIR}:rw" \
    --user root \
    -w /opt/moxa/dlm \
    ${REGISTRY_HOST}/openssl:${OPENSSL_VERSION} "$@"
}

source_environment_variables() {
  # get the directory of the script
  install_dir=$(dirname "$(dirname "$(readlink -f "$0")")")
  export install_dir
  if [ -f "${install_dir}/static-config/env-files/manifest.env" ]; then
    # shellcheck disable=SC2046
    export $(grep -v '^#' "${install_dir}/static-config/env-files/manifest.env" | xargs)
  fi
  if [ -f "${USER_DATA_DIR}/env-files/global.env" ]; then
    # shellcheck disable=SC2046
    export $(grep -v '^#' "${USER_DATA_DIR}/env-files/global.env" | xargs)
  fi
}

create_cert() {
  local type=${1:-"leaf"} # root-ca, intermediate, leaf
  local issuer=${2:-none} # none, root, {intermediate_name}
  local name=${3:-web}    # web, pic, dm, lwm2m
  local common_name=${4:-"Web Service"}
  local dir="${USER_DATA_DIR}/data/certs/${name}"
  local env_dir="${USER_DATA_DIR}/env-files/certs"
  cd "/opt/moxa/dlm" || exit 1

  SERVICE_FQDN=$(/opt/moxa/dlm/bin/dlm config global SERVICE_FQDN)
  SERVICE_NAME=$(/opt/moxa/dlm/bin/dlm config global SERVICE_NAME)
  # 检查操作系统并相应地获取 IP 地址
  if [[ "$OSTYPE" == "darwin"* ]]; then
    # macOS
    SERVICE_IPS=$(ifconfig | grep "inet " | grep -v 127.0.0.1 | awk '{print $2}')
  else
    # Linux 和其他系统
    SERVICE_IPS=$(hostname -I) || true
  fi
  mkdir -p "$dir" "$env_dir"
  # create private key
  if [ ! -f "${dir}/privkey.pem" ]; then
    touch "${dir}/privkey.pem"
    func_docker_openssl genrsa -traditional -out "${dir}/privkey.pem" 2048
  fi

  case $type in
  root-ca)
    # create root cert
    if [ ! -f "${dir}/cert.pem" ]; then
      touch "${dir}/cert.pem"
      func_docker_openssl req \
        -new \
        -nodes \
        -x509 \
        -key "${dir}/privkey.pem" \
        -out "${dir}/cert.pem" \
        -days 18250 \
        -sha256 \
        -subj "/CN=${common_name}"
      cp "${dir}/cert.pem" "${dir}/fullchain.pem"
    else
      echo "[warning] Root CA already exists, skip creating..."
    fi

    ;;
  intermediate)
    # create csr for intermediate cert
    if [ ! -f "${dir}/csr.pem" ]; then

      func_docker_openssl req \
        -new \
        -key "${dir}/privkey.pem" \
        -out "${dir}/csr.pem" \
        -subj "/CN=${common_name}"
    else
      echo "[warning] Intermediate CSR already exists, skip creating..."
    fi

    # create extfile.cnf for intermediate cert
    ip_string=""
    for ip in $SERVICE_IPS; do
      ip_string="${ip_string}IP:${ip},"
    done
    if [ -z "$ip_string" ]; then
      ip_string="IP: 127.0.0.1"
    fi
    ip_string=${ip_string%,}
    touch "${dir}/extfile.cnf"
    {
      echo "[v3_ca]"
      echo "basicConstraints = critical,CA:TRUE"
      echo "keyUsage = critical, cRLSign, digitalSignature, keyCertSign"
      echo "extendedKeyUsage = clientAuth, serverAuth"
      echo "subjectAltName=DNS:${SERVICE_FQDN},DNS:${SERVICE_NAME},${ip_string}"
    } | tee -a "${dir}/extfile.cnf" >/dev/null

    # create intermediate cert
    if [ ! -f "${dir}/cert.pem" ]; then
      touch "${dir}/cert.pem"
      func_docker_openssl x509 \
        -req \
        -in "${dir}/csr.pem" \
        -CA "${dir}/../${issuer}/cert.pem" \
        -CAkey "${dir}/../${issuer}/privkey.pem" \
        -extfile "${dir}/extfile.cnf" \
        -CAcreateserial \
        -extensions v3_ca \
        -out "${dir}/cert.pem" \
        -days 18250 \
        -sha256
      # create fullchain
      {
        cat "${dir}/cert.pem"
        cat "${dir}/../${issuer}/cert.pem"
      } | tee "${dir}/fullchain.pem" >/dev/null
    fi

    ;;
  leaf)
    # create csr for leaf cert
    if [ ! -f "${dir}/csr.pem" ]; then

      func_docker_openssl req \
        -new \
        -key "${dir}/privkey.pem" \
        -out "${dir}/csr.pem" \
        -subj "/CN=${common_name}"
    else
      echo "[warning] Leaf CSR already exists, skip creating..."
    fi

    # create extfile.cnf for leaf cert
    ip_string=""
    for ip in $SERVICE_IPS; do
      ip_string="${ip_string}IP:${ip},"
    done
    if [ -z "$ip_string" ]; then
      ip_string="IP: 127.0.0.1"
    fi
    ip_string=${ip_string%,}
    {
      echo "[v3_leaf]"
      echo "basicConstraints = critical, CA:FALSE"
      echo "keyUsage = critical, nonRepudiation, digitalSignature, keyEncipherment, keyAgreement"
      echo "extendedKeyUsage = critical, clientAuth, serverAuth"
      echo "subjectAltName=DNS:${SERVICE_FQDN},DNS:${SERVICE_NAME},${ip_string}"
    } | tee -a "${dir}/extfile.cnf" >/dev/null

    # create leaf cert
    if [ ! -f "${dir}/cert.pem" ]; then
      touch "${dir}/cert.pem"
      func_docker_openssl x509 \
        -req \
        -in "${dir}/csr.pem" \
        -CA "${dir}/../${issuer}/cert.pem" \
        -CAkey "${dir}/../${issuer}/privkey.pem" \
        -CAcreateserial \
        -out "${dir}/cert.pem" \
        -days 18250 \
        -sha256 \
        -extfile "${dir}/extfile.cnf" \
        -extensions v3_leaf

      # create fullchain
      {
        cat "${dir}/cert.pem"
        cat "${dir}/../${issuer}/cert.pem"
        cat "${dir}/../root/cert.pem"
      } | tee "${dir}/fullchain.pem" >/dev/null
    fi

    ;;
  *)
    echo "[error] Unknown type: $type"
    exit 1
    ;;
  esac

  # generate env file
  fullchain=$(cat "${dir}/fullchain.pem")
  cert=$(cat "${dir}/cert.pem")
  pk=$(cat "${dir}/privkey.pem")
  {
    echo "X509_MOXA_DLMG2_${name^^}_FULLCHAIN_CERT=${fullchain//$'\n'/\\n}"
    echo
    echo "X509_MOXA_DLMG2_${name^^}_CERT=${cert//$'\n'/\\n}"
    echo
    echo "X509_MOXA_DLMG2_${name^^}_KEY=${pk//$'\n'/\\n}"
  } | tee "${env_dir}/${name}.env" >/dev/null
  chmod -R 777 "${dir}/"
  cd - >/dev/null 2>&1 || true
}

create_dhparam() {
  local dir="${USER_DATA_DIR}/data/certs/web"

  mkdir -p "$dir"
  cd "$dir" || exit 1

  if [ ! -f "${dir}/dhparam.pem" ]; then
    func_docker_openssl dhparam -out "${dir}/dhparam.pem" 2048
  fi
  chmod -R 777 "${dir}/"
  cd - >/dev/null 2>&1 || true
}

up() {
  echo "Create certificates..."
  # use debconf-apt tool to show progress bar

  create_cert root-ca none root "DLM Root CA"
  create_cert intermediate root pic "DLM Provision Service"
  create_cert intermediate root dm "DLM Device Management Service"
  create_cert leaf root web "DLM Web Service"
  create_cert leaf pic lwm2m "DLM lightweight M2M Service"

  echo "Create dhparam.pem..."
  create_dhparam
}

down() {
  echo "Remove certificates..."
  rm -rf "${USER_DATA_DIR}/data/certs"
  rm -rf "${USER_DATA_DIR}/env-files/certs"
}

main() {
  arg1=${1:-usage}

  case $arg1 in
  up)
    up
    ;;
  down)
    down
    ;;
  *)
    echo "Usage: $0 {up|down}"
    exit 1
    ;;
  esac
}

source_environment_variables
main "$@"
